package logwire.web.bo.loader;

import logwire.core.bo.annotation.BizModel;
import logwire.core.bo.annotation.Column;
import logwire.core.bo.object.BizObject;
import logwire.core.exceptions.ApplicationException;
import logwire.core.meta.model.Model;
import logwire.core.meta.query.IQuery;
import logwire.core.resource.BeanLoader;
import logwire.core.resource.loader.JavaModelBeanLoader;
import logwire.core.tenant.TenantClassLoader;
import logwire.core.tenant.TenantProject;
import logwire.web.bo.field.parse.FieldParser;
import logwire.web.bo.field.parse.FieldParserUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.core.io.DefaultResourceLoader;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

public class BoModelBeanLoader extends BeanLoader<IQuery, TenantProject> implements JavaModelBeanLoader {

    @Autowired
    LoaderUtil loaderUtil;

    /**
     * load BoModel 的逻辑：
     * 1、先遍历出所有BizObject的子类
     * 2、将直接继承BizObject的类初始化
     * 3、剩余的类中，判断是否独立的model(包含BizModel注解)，若是，生成Model；若不是，字段添加至父类的Model中
     *
     * @param project  当前租户
     * @param input    待加载的资源
     * @param consumer 处理加载结果
     * @throws Exception
     */
    @Override
    public void load(TenantProject project, TenantProject input, Consumer<IQuery> consumer) throws Exception {
        TenantClassLoader classLoader = project.getTenantBeanContext().getTenantClassLoader();
        ClassPathScanningCandidateComponentProvider provider = loaderUtil.getScanningBeanProvider();
        provider.setResourceLoader(new DefaultResourceLoader(classLoader));
        provider.addIncludeFilter(loaderUtil.getAbstractTypeFilter(BizObject.class));
        Set<BeanDefinition> beans = provider.findCandidateComponents("logwire.web.bo");

        List<BeanDefinition> bizModelList = beans.stream().filter(beanDefinition -> {
            try {
                Class<?> clazz = classLoader.loadClass(beanDefinition.getBeanClassName());
                BizModel bizModelAnnotation = clazz.getAnnotation(BizModel.class);
                return bizModelAnnotation != null;
            } catch (ClassNotFoundException e) {
                throw new ApplicationException("");
            }
        }).collect(Collectors.toList());

        for (BeanDefinition beanDefinition : bizModelList) {
            Class<?> clazz = classLoader.loadClass(beanDefinition.getBeanClassName());
            //独立model
            Model model = new Model();
            model.setName(clazz.getSimpleName());
            model.setVerboseName(clazz.getAnnotation(BizModel.class).label());
            model.setIncludeAuditFields(true);
            model.setIncludeVersionField(true);
            model.setIncludeDomainField(true);
            initField(clazz, model);
           /*
            //todo
           this.initIndex(clazz, model);
            this.initSign(model);*/

            consumer.accept(model);

        }


    }

    private void initField(Class<?> clazz, Model model) {
        //获取所有带有Column注解的字段
        Field[] fields = clazz.getDeclaredFields();
        for (Field f : fields) {
            Column column = f.getAnnotation(Column.class);
            if (column != null) {
                Annotation[] annotations = f.getDeclaredAnnotations();
                FieldParser parser = FieldParserUtil.getParser(f, annotations);
                if (parser == null) {
                    continue;
                }
                logwire.core.meta.model.fields.Field field = parser.parse(f, annotations);
                model.addField(field);
            }
        }
    }

    @Override
    public boolean accept(TenantProject input) {
        return true;
    }
}
