package logwire.web.bo.loader;

import com.google.common.collect.Maps;
import logwire.core.bo.annotation.BizModel;
import logwire.core.bo.annotation.ListOperationProvider;
import logwire.core.bo.annotation.ObjectOperationProvider;
import logwire.core.bo.annotation.TypeOperationProvider;
import logwire.core.bo.eventhandler.ListOperationEventHandler;
import logwire.core.bo.eventhandler.ObjectOperationEventHandler;
import logwire.core.bo.eventhandler.OperationEventHandler;
import logwire.core.bo.eventhandler.TypeOperationEventHandler;
import logwire.core.bo.handler.ListOperationHandler;
import logwire.core.bo.handler.ObjectOperationHandler;
import logwire.core.bo.handler.OperationHandler;
import logwire.core.bo.handler.TypeOperationHandler;
import logwire.core.bo.object.BizObject;
import logwire.core.resource.BeanLoader;
import logwire.core.tenant.TenantClassLoader;
import logwire.core.tenant.TenantProject;
import logwire.web.bo.BoRelationDefinition;
import logwire.web.bo.proxy.Proxy;
import logwire.web.service.query.QueryService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.context.annotation.ScannedGenericBeanDefinition;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.type.filter.AnnotationTypeFilter;
import org.springframework.core.type.filter.AssignableTypeFilter;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

@Component
public class BoRelationBeanDefinitionLoader extends BeanLoader<BoRelationDefinition, TenantProject> {

    @Autowired
    QueryService queryService;

    @Autowired
    LoaderUtil loaderUtil;

    @Override
    public void load(TenantProject project, TenantProject input, Consumer<BoRelationDefinition> consumer) throws Exception {
        Map<String, BoRelationDefinition> boRelationBeanDefinitions = loadBeanDefinition(project);
        boRelationBeanDefinitions.values().forEach(boRelationDefinition -> {
            initializeService(boRelationDefinition, boRelationBeanDefinitions);
        });
    }

    /**
     * 初始化
     * 需先初始化 直接继承BizObject的BO的service
     *
     * @param boRelationDefinition
     * @param boRelationBeanDefinitions
     */
    public void initializeService(BoRelationDefinition boRelationDefinition, Map<String, BoRelationDefinition> boRelationBeanDefinitions) {
        if (!boRelationDefinition.isBoServiceInitialized()) {
            Class superClass = boRelationDefinition.getBoClass().getSuperclass();
            if (superClass != BizObject.class) {
                //若非直接继承 先初始化父类
                initializeService(boRelationBeanDefinitions.get(superClass.getSimpleName()), boRelationBeanDefinitions);
            }
            try {
                //create proxy typeOperationProvider
                Object o = Proxy.createTypeOperationProvider(boRelationDefinition);
                Field field = boRelationDefinition.getBoClass().getField("service");
                field.setAccessible(true);
                field.set(null, o);
                boRelationDefinition.setBoServiceInitialized(true);
            } catch (NoSuchFieldException | IllegalAccessException | InstantiationException e) {
                //TODO
                e.printStackTrace();
            }
        }
    }

    /**
     * @param project
     * @return
     * @throws Exception
     */
    private Map<String, BoRelationDefinition> loadBeanDefinition(TenantProject project) throws Exception {
        Map<String, BoRelationDefinition> boRelationBeanDefinitions = Maps.newHashMap();
        TenantClassLoader classLoader = project.getTenantBeanContext().getTenantClassLoader();
        ClassPathScanningCandidateComponentProvider provider = loaderUtil.getScanningBeanProvider();
        provider.setResourceLoader(new DefaultResourceLoader(classLoader));
        addIncludeFilters(provider);
        Set<BeanDefinition> boClassDefinitions = provider.findCandidateComponents("logwire.web.bo");
        //beans 是有序的
        for (BeanDefinition boClassDefinition : boClassDefinitions) {
            Class<?> clazz = Class.forName(boClassDefinition.getBeanClassName(), true, classLoader);
            if (loaderUtil.isBizObject(clazz)) {
                BoRelationDefinition boRelationDefinition = doRegisterBeanDefinition((Class<? extends BizObject>) clazz);
                boRelationBeanDefinitions.put(boRelationDefinition.getId(), boRelationDefinition);
            } else {
                //handler
                if (loaderUtil.isOperationHandler(clazz)) {
                    if (!isConcrete((ScannedGenericBeanDefinition) boClassDefinition)) continue;
                    addOperationHandler(boRelationBeanDefinitions, (Class<OperationHandler>) clazz);
                }
                //event_handler
                else if (loaderUtil.isOperationEventHandler(clazz)) {
                    if (!isConcrete((ScannedGenericBeanDefinition) boClassDefinition)) continue;
                    addOperationEventHandler(boRelationBeanDefinitions, (Class<OperationEventHandler>) clazz);
                }
                //operation
                else if (clazz.getAnnotation(TypeOperationProvider.class) != null) {
                    Class<? extends BizObject> type = clazz.getAnnotation(TypeOperationProvider.class).type();
                    boRelationBeanDefinitions.get(type.getSimpleName()).addTypeOperationProvider(clazz);
                } else if (clazz.getAnnotation(ObjectOperationProvider.class) != null) {
                    Class<? extends BizObject> type = clazz.getAnnotation(ObjectOperationProvider.class).type();
                    boRelationBeanDefinitions.get(type.getSimpleName()).addObjectOperationProvider(clazz);
                } else if (clazz.getAnnotation(ListOperationProvider.class) != null) {
                    Class<? extends BizObject> type = clazz.getAnnotation(ListOperationProvider.class).type();
                    boRelationBeanDefinitions.get(type.getSimpleName()).addListOperationProvider(clazz);
                }
            }
        }
        return boRelationBeanDefinitions;
    }


    private BoRelationDefinition doRegisterBeanDefinition(Class<? extends BizObject> beanClass) {
        BoRelationDefinition boRelationDefinition = new BoRelationDefinition();
        boRelationDefinition.setId(beanClass.getSimpleName());
        boRelationDefinition.setName(beanClass.getSimpleName());
        boRelationDefinition.setBoClass(beanClass);
        boRelationDefinition.setBizModel(beanClass.isAnnotationPresent(BizModel.class));
        return boRelationDefinition;
    }


    private void addIncludeFilters(ClassPathScanningCandidateComponentProvider provider) {
        provider.addIncludeFilter(loaderUtil.getAbstractTypeFilter(BizObject.class));
        provider.addIncludeFilter(new AssignableTypeFilter(OperationHandler.class));
        provider.addIncludeFilter(new AssignableTypeFilter(OperationEventHandler.class));
        provider.addIncludeFilter(new AnnotationTypeFilter(TypeOperationProvider.class));
        provider.addIncludeFilter(new AnnotationTypeFilter(ObjectOperationProvider.class));
        provider.addIncludeFilter(new AnnotationTypeFilter(ListOperationProvider.class));
    }

    private void addOperationEventHandler(Map<String, BoRelationDefinition> bizObjectBeanMap, Class<OperationEventHandler> clazz) throws IllegalAccessException, InstantiationException {
        //get BizObject from genericInterface
        Class<? extends BizObject> bizObject = (Class<? extends BizObject>) ((ParameterizedType) clazz.getGenericInterfaces()[0]).getActualTypeArguments()[0];
        BoRelationDefinition boRelationDefinition = bizObjectBeanMap.get(bizObject.getSimpleName());
        OperationEventHandler opEventHandler = clazz.newInstance();
        if (opEventHandler.isEnabled()) {
            if (TypeOperationEventHandler.class.isAssignableFrom(clazz)) {
                boRelationDefinition.getTypeOperationEventHandlers().add((TypeOperationEventHandler) opEventHandler);
            } else if (TypeOperationHandler.class.isAssignableFrom(clazz)) {
                boRelationDefinition.getObjectOperationEventHandlers().add((ObjectOperationEventHandler) opEventHandler);
            } else {
                boRelationDefinition.getListOperationEventHandlers().add((ListOperationEventHandler) opEventHandler);
            }
        }
    }

    private void addOperationHandler(Map<String, BoRelationDefinition> bizObjectBeanMap, Class<OperationHandler> clazz) throws IllegalAccessException, InstantiationException {
        //get BizObject from genericInterface
        Class<? extends BizObject> bizObject = (Class<? extends BizObject>) ((ParameterizedType) clazz.getGenericInterfaces()[0]).getActualTypeArguments()[0];
        BoRelationDefinition boRelationDefinition = bizObjectBeanMap.get(bizObject.getSimpleName());
        OperationHandler opHandler = clazz.newInstance();
        if (opHandler.isEnabled()) {
            if (TypeOperationHandler.class.isAssignableFrom(clazz)) {
                boRelationDefinition.getTypeOperationHandlers().add((TypeOperationHandler) opHandler);
            } else if (ObjectOperationHandler.class.isAssignableFrom(clazz)) {
                boRelationDefinition.getObjectOperationHandlers().add((ObjectOperationHandler) opHandler);
            } else if (ListOperationHandler.class.isAssignableFrom(clazz)) {
                boRelationDefinition.getListOperationHandlers().add((ListOperationHandler) opHandler);
            }
        }
    }

    private boolean isConcrete(ScannedGenericBeanDefinition beanDefinition) {
        return beanDefinition.getMetadata().isConcrete();
    }

    @Override
    public boolean accept(TenantProject input) {
        return true;
    }
}
