import Dockerode from "dockerode"
import LogUtil, { sendSocketMessage } from "~/server/utils/log"
import path from 'path'

export default defineEventHandler(async (event) => {
  let username = event.context.username
  let isClonedSuccess = true
  let sshKey = ''
  try {
    LogUtil.clear(username)
    let docker = createDockerFactory(username)
    let rowV1 = await getPgTableData('v1', username)
    if (!rowV1) await insertPgTableData('v1', username)
    await changeProjectStatus( 'v1', username, 'status', 'creating')
    let container = await docker.checkContainer( username + '.node') as Dockerode.Container
    if (!container) {
      throw new Error('没有创建容器，请先初始化容器')
    }
    // 创建 postgres 不一定要在本机，即使是 dev 环境也可以联远程的容器
    await LogUtil.run(username, 'v1-创建 postgres 容器', async () => {
      if (process.env.NODE_ENV?.trim() !== 'production') {
        let postgresInfo = await docker.checkContainer('postgres_12')
        if (!postgresInfo) {
          let postgres = await docker.checkAndCreateContainer({ name: 'postgres_12', img: 'postgres:12', env: ["POSTGRES_PASSWORD=postgres"], portBindings: { '5432/tcp': [{ HostPort: PgPort.toString() }] } })
          await docker.startContainer({ container: postgres })
          await sleep(3000)
        }

        const database = username + '_v1'
        let client = await createPgClientFactory()
        let result = await executePgQuery({ client, query: "SELECT u.datname  FROM pg_catalog.pg_database u where u.datname='" + database + "';" })
        if (result.rows.length === 0) {
          await executePgQuery({ client, query: 'CREATE DATABASE ' + database })
          await stopPgClient(client)
          client = await createPgClient({ host: HOST, port: +PgPort, database: database })
          await executePgQuery({ client, query: 'CREATE SCHEMA library' })
          await stopPgClient(client)
        }
      }
    })

    await LogUtil.run(username, 'v1-创建 redis 容器', async () => {
      if (process.env.NODE_ENV?.trim() !== 'production') {
        let redisInfo = await docker.checkContainer('redis')
        if (!redisInfo) {
          let redis = await docker.checkAndCreateContainer({ name: 'redis', img: 'redis', portBindings: { '6379/tcp': [{ HostPort: '6379' }] } })
          await docker.startContainer({ container: redis })
        }
      }
    })

    await LogUtil.run(username, '检查 SSH Key', async () => {
      sshKey = await checkSshExists(username)
      LogUtil.printWarning(username, `${username}@greaconsulting.com 用户 SSH key 为 <strong>${sshKey.replace(/\n/g, '')}</strong>, 若克隆仓库失败, 请检查是否放入 Gitlab SSH 设置中并重试\n`)
    })

    await LogUtil.run(username, 'v1-克隆仓库', async () => {
      isClonedSuccess = false
      await docker.execContainerCommand({ container, cmd: ['git', 'config' ,'--global', 'core.sshCommand', 'ssh -o UserKnownHostsFile=/dev/null -o StrictHostKeyChecking=no'] })
      await docker.execContainerCommand({ container, cmd: 'rm -rf /var/logwire-platform' })
      await docker.execContainerCommand({ container, cmd: 'git clone ssh://git@gitlab.logwire.cn:13389/logwire/logwire-platform.git', dir: '/var' })
      isClonedSuccess = true
    })

    await LogUtil.run(username, 'v1-安装 openjdk ', async () => {
      // 文件过大，已存储到容器 /var 目录下
      await docker.putArchive({ container, tarPath: path.resolve('/var/openlogic-openjdk-8u402-b06-linux-x64.tar.gz'), targetPath: '/var' })
      await docker.execContainerCommand({ container, cmd: 'cp -r openlogic-openjdk-8u402-b06-linux-x64/ java-8-openjdk', dir: '/var' })
      await docker.execContainerCommand({ container, cmd: 'rm -rf openlogic-openjdk-8u402-b06-linux-x64', dir: '/var' })
    })

    await LogUtil.run(username, 'v1-下载字符集', async () => {
      await docker.execContainerCommand({ container, cmd: 'rm -f /etc/locale.gen'})
      await docker.execContainerCommand({ container, cmd: 'touch /etc/locale.gen' })
      await docker.writeFile({ container, path: '/etc/locale.gen', text: '\'\nzh_CN.UTF-8 UTF-8\''})
      await docker.execContainerCommand({ container, cmd: 'apt-get install -y locales' })
      await docker.execContainerCommand({ container, cmd: 'locale-gen', dir: '/etc'})
    })

    await LogUtil.run(username, 'v1-迁移logwire-demo', async () => {
      await docker.execContainerCommand({ container, cmd: 'rm -rf /var/logwire-demo' })
      await docker.execContainerCommand({ container, cmd: 'mkdir logwire-demo', dir: '/var'})
      await docker.execContainerCommand({ container, cmd: 'mkdir -p dist/projects/demo', dir: '/var/logwire-platform'})
      // 避免上传太大的压缩文件，需要先在 node 服务器的 /var 目录下，上传该文件
      await docker.putArchive({ container, tarPath: path.resolve('/var/book.tar'), targetPath: '/var/logwire-demo' })
      await docker.execContainerCommand({ container, cmd: 'tar -zvcf ../book.tar .', dir: '/var/logwire-demo/book'})
      await docker.execContainerCommand({ container, cmd: 'tar -zvxf book.tar -C /var/logwire-platform/dist/projects/demo', dir: '/var/logwire-demo'})
    })

    await changeProjectStatus( 'v1', username, 'status', 'created')
    LogUtil.printSuccess(username, 'installed')
  } catch (err: any) {
    await changeProjectStatus( 'v1', username, 'status', 'null')
    if (!isClonedSuccess) {
      const message = `初始化中克隆仓库失败，请将 <strong>${sshKey.replace(/\n/g, '')}</strong> 放入 Gitlab SSH 设置中并重试`
      const type = 'Tip'
      const eventData = { type, message }
      sendSocketMessage(username, 'v1', JSON.stringify(eventData))
    }
    LogUtil.printError(username, (err instanceof Error ? err.message : JSON.stringify(err)))
    setResponseStatus(event, 500)
    return err
  }
})